/**
 * \file: libsugc_test.c
 *
 * \brief : Functions to read the configuration
 * Note: the glib key value parser is calling the different sections in a config
 *       groups - in order not to get confused with group-name, group-id etc.
 *       these glib key-value parser groups are named sections.
 *
 * \author: Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2017 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <libsugc.h>
#include <unistd.h>
#include <semaphore.h>
#include <errno.h>
#include <signal.h>

sem_t wait_for_term_signal;

#define ERROR(msgformat,...) fprintf(stderr,msgformat"\n", ## __VA_ARGS__)

static void term_signal_handler(int sig) {
    int res;
    (void)sig;

    errno = 0;
    res = sem_post(&wait_for_term_signal);
    if (res != 0) {
        ERROR("sem post failed %d/%s",
              res, strerror(errno));
    }
}

int main(int argc, char **argv)
{
    libsugc_error_t err;
    int res;

    if (argc != 2) {
        printf("Usage: %s <CONFIG_NAME>\n", argv[0]);
        exit(2); /* don't use default error value */
    }

    errno = 0;
    res = sem_init(&wait_for_term_signal, 0, 0);
    if (res != 0) {
        ERROR("Failed to init semaphore %d/%s",
              res, strerror(errno));
        return EXIT_FAILURE;
    }

    errno = 0;
    if (SIG_ERR == signal(SIGTERM, term_signal_handler)) {
        ERROR("Failed to register term signal handler %s",
              strerror(errno));
    }

    /* apply the actual test configuration */
    err = libsugc_apply_cfg(argv[1]);

    res = -1;
    while (res != 0)
    {
        errno = 0;
        res = sem_wait(&wait_for_term_signal);
        if (res != 0) {
            ERROR("sem post failed %d/%s",
                  res, strerror(errno));
        }
    }

    return (int)err;
}
